//--------------------------------------------------------------------------------------
// File: Demo.cpp
//
// This application demonstrates creating a Direct3D 11 device
//
// Copyright (c) Microsoft Corporation. All rights reserved.
//--------------------------------------------------------------------------------------


#include "resource.h"

#include "Blobs/public.h"

#define NOMINMAX

#include <time.h>
#include <comdef.h> 
#include <windows.h>
#include <d3d11.h>
#include <d3dx11.h>

#ifdef PlaySound
#undef PlaySound
#endif

#include <vector>
#include <type_traits>
#include <xnamath.h>

#include "TimeManager.h"
#include "SoundManager.h"
#include "SequenceManager.h"
#include "ConstantBuffers.h"
#include "HelperShader.h"
#include "SequenceRenderer.h"

#include "Animation.h"

#define VK_R 0x52
#define VK_r 0x72

//--------------------------------------------------------------------------------------
// Structures
//--------------------------------------------------------------------------------------
struct SimpleVertex
{
    XMFLOAT3 Pos;
    XMFLOAT2 Tex;
    XMFLOAT3 Nrm;
};

typedef std::vector<SimpleVertex> SimpleVertexArray;

//--------------------------------------------------------------------------------------
// Global Variables
//--------------------------------------------------------------------------------------
HINSTANCE					g_hInst				            = NULL;
HWND						g_hWnd				            = NULL;
D3D_DRIVER_TYPE				g_driverType		            = D3D_DRIVER_TYPE_NULL;
D3D_FEATURE_LEVEL			g_featureLevel		            = D3D_FEATURE_LEVEL_11_0;
ID3D11Device*				g_pd3dDevice		            = NULL;
ID3D11DeviceContext*		g_pImmediateContext             = NULL;
IDXGISwapChain*				g_pSwapChain		            = NULL;
ID3D11RenderTargetView*		g_pRenderTargetView             = NULL;
ID3D11VertexShader*			g_pVertexShader		            = NULL;
ID3D11PixelShader*			g_pPixelShader		            = NULL;
ID3D11InputLayout*			g_pVertexLayout		            = NULL;
ID3D11Buffer*				g_pScreenQuadVertexBuffer		= NULL;
ID3D11Buffer*				g_pTriangleVertexBuffer		    = NULL;
ID3D11ShaderResourceView*   g_pTextureRV		            = NULL;
ID3D11SamplerState*         g_pSamplerLinear	            = NULL;
ID3D11Buffer*               g_pCBNeverChanges               = NULL;
ID3D11Buffer*               g_pCBChangesEveryFrame          = NULL;
CBLedPorperties             g_CBLedPorperties;
ID3D11Buffer*               g_pCBLedPorperties              = NULL;
XMMATRIX                    g_World                         = XMMatrixIdentity();
XMMATRIX                    g_View                          = XMMatrixIdentity();
XMMATRIX                    g_Projection                    = XMMatrixIdentity();
TimeManager*                g_pTimeManager                  = NULL;
SequenceManager*            g_pSequenceManager              = NULL;
SequenceRenderer*           g_pSequenceRenderer             = NULL;
SoundManager*               g_pSoundManager                 = NULL;
// should be moved to something like "TextureTransitionVS.hlsl" and "TextureTransitionPS.hlsl".
// see http://social.msdn.microsoft.com/Forums/en-US/wingameswithdirectx/thread/51859322-fc36-4946-b4cb-b5971fcaa9e5
const LONG              g_screenWidth       = 1024;
const LONG              g_screenHeigth      = 768;
const float             g_energyThreshold   = 0.35f;
const float             g_energyThresholdInv= 2.857142857f;
static const int        LedCubeSide = 16;
static const int        LedCubeSideMinusOne =  LedCubeSide - 1;
static const int        LedCount = LedCubeSide * LedCubeSide * LedCubeSide;
static const int        LedCountMinusOne = LedCount - 1;
static const float      HalfLedCubeSide = 0.5*LedCubeSide;

//todo : refactor
unsigned int		    g_uiQuadVertexCount         = 0;
unsigned int		    g_uiTriangleVertexCount     = 0;

const CBNeverChanges    g_cbNeverChange     = {45.0f, (float)g_screenWidth / (float)g_screenHeigth };




//--------------------------------------------------------------------------------------
// Metaballs stuff
//--------------------------------------------------------------------------------------
using namespace V3D;

Vect3DArray                                    g_aBlobVertices;
Vect3fArray                                    g_aBlobNormals;
TriangleArray                                  g_aBlobTriangles;
IsoSurfaceBlobsObject::BlobsInfluencesIndices  g_aBlobInfluencesIdx;
IsoSurfaceBlobsObject::BlobsInfluencesValues   g_aBlobInfluencesVal;

SimpleVertexArray  g_aBlobSimpleVertices;

BlobEllipsoid g_blob1(100.f, Vector3D(0,0,0), 30.f);
BlobEllipsoid g_blob2(100.f, Vector3D(0,0,0), 30.f);
BlobEllipsoid g_blob3(100.f, Vector3D(0,0,0), 30.f);
BlobEllipsoid g_blob4(100.f, Vector3D(0,0,0), 30.f);
BlobEllipsoid g_blob5(100.f, Vector3D(0,0,0), 30.f);
BlobEllipsoid g_blob6(100.f, Vector3D(0,0,0), 30.f);

IsoSurfaceBlobsObject g_objBlobs = IsoSurfaceBlobsObject(30.f);
int GRIDFACTOR = 5;
MarchCubesByLayers g_metaballsTriangulator = MarchCubesByLayers(Vector3D(0.f, 0.f, 0.f)*float(GRIDFACTOR), Vector3D(130.f, 130.f, 130.f)*float(GRIDFACTOR), 32*GRIDFACTOR, 32*GRIDFACTOR, 32*GRIDFACTOR);

#define M_PI 3.14159265358979323846

// Formules du mouvement recuperes des sources de la demo State Of Mind
// fCurTime est attendu en secondes, je crois
void UpdateBlobPositions_StateOfMind(float fCurTime)
{
    g_objBlobs.RemoveAllBlobs();
    g_objBlobs.AddBlob( g_blob1 );
    g_objBlobs.AddBlob( g_blob2 );
    g_objBlobs.AddBlob( g_blob3 );
    g_objBlobs.AddBlob( g_blob4 );
    g_objBlobs.AddBlob( g_blob5 );

    //Vector3D vcSizeGrid   = Vector3D(3.1f, 3.1f, 3.1f) * float(GRIDFACTOR);
    Vector3D vcSizeGrid   = Vector3D(3.1f, 3.1f, 3.1f);
    Vector3D vcCornerGrid = Vector3D(0,0,0);
    //g_metaballsTriangulator.InitGrid(vcCornerGrid, vcSizeGrid, 32*GRIDFACTOR,32*GRIDFACTOR,32*GRIDFACTOR );
    g_metaballsTriangulator.InitGrid(vcCornerGrid, vcSizeGrid, 16,16,16 );
    g_objBlobs.SetEnergyThresh(g_energyThreshold);

    float x = (float) fmod((fCurTime * 0.8f),1.f);
    float y = (fCurTime * 0.1f);
    //	float y = (float) fmod((fCurTime * 0.0002f),1.f);

    x = (float)cos( x*2.0*M_PI );
    x = x * x * x * x;

    float R = 0.5f + 0.69f*x;
    float B_Rot = y * 1500.f;

    const float M_PI_180 = M_PI / 180.f;


    Vector3D vecBlob0( R * .66f * (float)sin( .243f * B_Rot * M_PI_180 ),
                       R * 0.6f * (float)cos( 1.9012f * B_Rot * M_PI_180 ),
                       R * 0.7f * (float)sin( .543f * B_Rot * M_PI_180+.2142f ) );

    Vector3D vecBlob1( R * .9212f * (float)sin( 0.7613f * B_Rot * M_PI_180 ),
                       0,
                       0 );

    Vector3D vecBlob2( 0,
                       0, 
                       R * 0.883f * (float)sin( 1.443f * B_Rot * M_PI_180 ) );

    Vector3D vecBlob3( 0,
                       R * 0.92151f * (float)cos( 1.64f * B_Rot * M_PI_180 ),
                       R * 0.9524f * (float)sin( 1.64f * B_Rot * M_PI_180 ) );

    Vector3D vecBlob4( R * 2 * 0.4f * (float)sin( 1.312f * B_Rot * M_PI_180 ),
                       0,
                       R * 2 * 0.3151f * (float)cos( .735f * B_Rot * M_PI_180+.124f ) );


    //Vector3D vcCentre(1.5f, 1.5f, 1.5f);
    g_blob1.SetInfluenceRange(.7f);
    g_blob2.SetInfluenceRange(.7f);
    g_blob3.SetInfluenceRange(.7f);
    g_blob4.SetInfluenceRange(.7f);
    g_blob5.SetInfluenceRange(.7f);

    g_blob1.SetPosition(vecBlob0);
    g_blob2.SetPosition(vecBlob1);
    g_blob3.SetPosition(vecBlob2);
    g_blob4.SetPosition(vecBlob3);
    g_blob5.SetPosition(vecBlob4);

    g_blob1.SetCenterEnergy(1.f);
    g_blob2.SetCenterEnergy(1.f);
    g_blob3.SetCenterEnergy(1.f);
    g_blob4.SetCenterEnergy(1.f);
    g_blob5.SetCenterEnergy(1.f); 
}


// This function makes the blobs mesh info into g_aBlobSimpleVertices and g_aBlobTriangles
// g_aBlobTriangles can be seen as an array of uint32 : (uint32*) &g_aBlobTriangles[0]
void MakeMetaballsMesh()
{
    g_objBlobs.GenerateMesh( g_metaballsTriangulator, g_aBlobVertices, g_aBlobTriangles, g_aBlobInfluencesIdx, &g_aBlobNormals, true );
    // Build SimpleVertex array from our arrays of positions and normals
    std::size_t nbVertices = g_aBlobVertices.size();
    assert( g_aBlobVertices.size() == g_aBlobNormals.size() );
    g_aBlobVertices.resize( nbVertices );
    g_aBlobSimpleVertices.resize( nbVertices );
    for( std::size_t i = 0; i < nbVertices; ++i )
    {
        SimpleVertex& simpleVtx = g_aBlobSimpleVertices[i];
        const Vector3D& pos = g_aBlobVertices[i];
        const Vector3f& nrm = g_aBlobNormals[i];
        simpleVtx.Pos = XMFLOAT3( pos.x, pos.y, pos.z );
        simpleVtx.Nrm = XMFLOAT3( nrm.x, nrm.y, nrm.z );
    }
}


//--------------------------------------------------------------------------------------
// Forward declarations
//--------------------------------------------------------------------------------------
HRESULT             InitWindow( HINSTANCE hInstance, int nCmdShow );
HRESULT             LoadResources();
HRESULT             InitDevice();
void                CleanupDevice();
void                Shutdown();
LRESULT CALLBACK    WndProc( HWND, UINT, WPARAM, LPARAM );
void                Render();
HRESULT             CreateVertexBuffer(const SimpleVertex* vertices, const size_t size, ID3D11Buffer** ppVertexBuffer);
HRESULT             CreateIndexBuffer(const Triangle* indexes, const size_t size, ID3D11Buffer** ppIndexBuffer) ;

void CALLBACK       OnKeyboard( UINT nChar, bool bKeyDown, bool bAltDown, void* pUserContext );


//--------------------------------------------------------------------------------------
// Entry point to the program. Initializes everything and goes into a message processing 
// loop. Idle time is used to render the scene.
//--------------------------------------------------------------------------------------
int WINAPI wWinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, LPWSTR lpCmdLine, int nCmdShow )
{
    UNREFERENCED_PARAMETER( hPrevInstance );
    UNREFERENCED_PARAMETER( lpCmdLine );

    //DXUTSetCallbackKeyboard( OnKeyboard );

    if( FAILED( InitWindow( hInstance, nCmdShow ) ) )
        return 0;

    if( FAILED( InitDevice() ) )
    {
        CleanupDevice();
        return 0;
    }

    int lastSequenceIndex = -1;
    bool hasPlayedAllSequences = false;
    // Main message loop
    MSG msg = {0};
    while( WM_QUIT != msg.message && !hasPlayedAllSequences )
    {
        if( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE ) )
        {
            TranslateMessage( &msg );
            DispatchMessage( &msg );

            switch(msg.message)
            {
            case WM_KEYUP:
                {
                    switch (msg.wParam)
                    {
                    case VK_ADD         : g_pTimeManager->ExecuteScaleCommand(ScaleFactorCommand::Increase); break;
                    case VK_SUBTRACT    : g_pTimeManager->ExecuteScaleCommand(ScaleFactorCommand::Decrease); break;
                    case VK_SPACE       : g_pTimeManager->ExecuteScaleCommand(ScaleFactorCommand::Reset); break;
                    case VK_R          :
                    case VK_r           : LoadResources(); break;
                    default:
                        break;
                    }
                    g_pSoundManager->SetScaleFactor( g_pTimeManager->GetScaleFactor() );
                }
            default: break;
            }
        }
        else
        {
            Render();
            g_pSoundManager->Update();
           int sequenceIndex = g_pSequenceManager->GetSequenceIndex(g_pTimeManager->GetTime(), g_pTimeManager->GetDeltaTime());

           if (sequenceIndex < lastSequenceIndex)
           {
               hasPlayedAllSequences = true;
           }

           lastSequenceIndex = sequenceIndex;
        }
    }

    CleanupDevice();
    Shutdown();

    return ( int )msg.wParam;
}


//--------------------------------------------------------------------------------------
// Register class and create window
//--------------------------------------------------------------------------------------
HRESULT InitWindow( HINSTANCE hInstance, int nCmdShow )
{
    // Register class
    WNDCLASSEX wcex;
    wcex.cbSize = sizeof( WNDCLASSEX );
    wcex.style = CS_HREDRAW | CS_VREDRAW;
    wcex.lpfnWndProc = WndProc;
    wcex.cbClsExtra = 0;
    wcex.cbWndExtra = 0;
    wcex.hInstance = hInstance;
    wcex.hIcon = LoadIcon( hInstance, ( LPCTSTR )IDI_TUTORIAL1 );
    wcex.hCursor = LoadCursor( NULL, IDC_ARROW );
    wcex.hbrBackground = ( HBRUSH )( COLOR_WINDOW + 1 );
    wcex.lpszMenuName = NULL;
    wcex.lpszClassName = L"Vortex - WIP";
    wcex.hIconSm = LoadIcon( wcex.hInstance, ( LPCTSTR )IDI_TUTORIAL1 );
    if( !RegisterClassEx( &wcex ) )
        return E_FAIL;

    // Create window
    g_hInst = hInstance;
    RECT rc = { 0, 0, g_screenWidth, g_screenHeigth };
    AdjustWindowRect( &rc, WS_OVERLAPPEDWINDOW, FALSE );
    g_hWnd = CreateWindow( L"Vortex - WIP", L"Vortex - WIP - Animatic", WS_OVERLAPPEDWINDOW,
                           CW_USEDEFAULT, CW_USEDEFAULT, rc.right - rc.left, rc.bottom - rc.top, NULL, NULL, hInstance,
                           NULL );
    if( !g_hWnd )
        return E_FAIL;

    ShowWindow( g_hWnd, nCmdShow );

    return S_OK;
}


//--------------------------------------------------------------------------------------
// Called every time the application receives a message
//--------------------------------------------------------------------------------------
LRESULT CALLBACK WndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
    PAINTSTRUCT ps;
    HDC hdc;

    switch( message )
    {
        case WM_PAINT:
            hdc = BeginPaint( hWnd, &ps );
            EndPaint( hWnd, &ps );
            break;

        case WM_DESTROY:
            PostQuitMessage( 0 );
            break;

        default:
            return DefWindowProc( hWnd, message, wParam, lParam );
    }

    return 0;
}

//--------------------------------------------------------------------------------------
// Load resources
//--------------------------------------------------------------------------------------
HRESULT LoadResources() 
{
    HRESULT hr = S_OK;
    std::vector<RenderingParameter>::iterator it =  g_pSequenceRenderer->GetParameters().begin();
    for ( const SequenceSeed& seed : g_pSequenceManager->GetSeeds() )
    {
        RenderingParameter& parameter = (*it);

        ID3DBlob* pVSBlob = NULL;
        if(seed.VSPath)
        {
          
            hr = HelperShader::CompileVShaderFromSource(_bstr_t(seed.VSPath), &pVSBlob); // dirty
            if( SUCCEEDED( hr ) )
            {
                hr = g_pd3dDevice->CreateVertexShader( pVSBlob->GetBufferPointer(), pVSBlob->GetBufferSize(), NULL, &parameter.pVertexShader);

                if( FAILED( hr ) )
                {	
                    pVSBlob->Release();
                    return hr;
                }

            }

        }

        if(seed.PSPath)
        {
            ID3D11PixelShader* pPreviousShader = parameter.pPixelShader;    
            ID3DBlob* pPSBlob = NULL;
            hr = HelperShader::CompilePShaderFromSource(_bstr_t(seed.PSPath), &pPSBlob); // dirty
            if( SUCCEEDED( hr ) )
            {
                hr = g_pd3dDevice->CreatePixelShader(pPSBlob->GetBufferPointer(), pPSBlob->GetBufferSize(), NULL, &parameter.pPixelShader);
                if( FAILED( hr ) )
                {	
                    pPSBlob->Release();
                    return hr;
                }  
                
                if(pPreviousShader)pPreviousShader->Release(); // to do : memorise and release on on sucessfull compilation
            }

          

        }

        if(seed.texturePath)
        {
            D3DX11CreateShaderResourceViewFromFile( g_pd3dDevice, _bstr_t(seed.texturePath), NULL, NULL, &parameter.pTextureRV, NULL );
        }

        switch (seed.meshType)
        {
        case MeshType::ScreenQuad :
            {
                
                SimpleVertex screenQuadVertices[] =
                {
                    {XMFLOAT3(-1.0f, 1.0f, 0.0f ),		XMFLOAT2( 0.0f, 0.0f ), XMFLOAT3(0.0f, 0.0f, 1.0f ) },
                    {XMFLOAT3( 1.0f, 1.0f, 0.0f ),		XMFLOAT2( 1.0f, 0.0f ), XMFLOAT3(0.0f, 0.0f, 1.0f ) },
                    {XMFLOAT3( -1.0f, -1.0f, 0.0f ),	XMFLOAT2( 0.0f, 1.0f ), XMFLOAT3(0.0f, 0.0f, 1.0f ) },
                    {XMFLOAT3( 1.0f, -1.0f, 0.0f ),		XMFLOAT2( 1.0f, 1.0f ), XMFLOAT3(0.0f, 0.0f, 1.0f ) },
                };

                // todo ? : wrapp this stuff in a util class ? 
                parameter.PrimitiveTopology = D3D_PRIMITIVE_TOPOLOGY::D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP;
                parameter.uiMeshVerticesCount = std::extent<decltype(screenQuadVertices)>::value;
                parameter.uiMeshStride = sizeof( SimpleVertex );
                parameter.uiMeshOffset = 0;
                
                hr = CreateVertexBuffer( screenQuadVertices, parameter.uiMeshVerticesCount, &parameter.pMeshVtxBuffer );
                if( FAILED( hr ) )
                    return hr;

                // copy pasted code as all meshtype share the same layout for now
                if(pVSBlob)
                {
                    // Define the input layout
                    D3D11_INPUT_ELEMENT_DESC layout[] =
                    {
                        { "POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0 },
                        { "TEXCOORD", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 12, D3D11_INPUT_PER_VERTEX_DATA, 0 },
                        { "NORMAL"  , 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, 20, D3D11_INPUT_PER_VERTEX_DATA, 0 },
                    };
                    UINT numElements = ARRAYSIZE( layout );

                    // Create the input layout
                    hr = g_pd3dDevice->CreateInputLayout(   layout, 
                        numElements, 
                        pVSBlob->GetBufferPointer(),
                        pVSBlob->GetBufferSize(), 
                        &parameter.pVertexLayout );

                    pVSBlob->Release();
                    pVSBlob = NULL;
                    if( FAILED( hr ) )
                        return hr;
                }
            }
            
            break;
        
        case MeshType::TestTriangle :
            {
                
                const float h = 0.86602540378443; // squareRoot(3)/2  
                SimpleVertex triangleVertices[] =
                {
                    {XMFLOAT3(0.0f   ,   0.5f *h    , 0.0f ),	XMFLOAT2( 0.0f, 0.0f ), XMFLOAT3(0.0f, 0.0f, 1.0f ) },
                    {XMFLOAT3(0.5f  ,    -0.5f *h   , 0.0f ),   XMFLOAT2( 1.0f, 0.0f ), XMFLOAT3(0.0f, 0.0f, 1.0f ) },
                    {XMFLOAT3(-0.5f ,    -0.5f *h   , 0.0f ),	XMFLOAT2( 0.0f, 1.0f ), XMFLOAT3(0.0f, 0.0f, 1.0f ) },
                };
                
                parameter.PrimitiveTopology = D3D_PRIMITIVE_TOPOLOGY::D3D11_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP;
                parameter.uiMeshVerticesCount = std::extent<decltype(triangleVertices)>::value;
                parameter.uiMeshStride = sizeof( SimpleVertex );
                parameter.uiMeshOffset = 0;
                
                hr = CreateVertexBuffer(triangleVertices,  parameter.uiMeshVerticesCount, &parameter.pMeshVtxBuffer);
                if( FAILED( hr ) )
                    return hr;
                
                // copy pasted code as all meshtype share the same layout for now
                if(pVSBlob)
                {
                    // Define the input layout
                    D3D11_INPUT_ELEMENT_DESC layout[] =
                    {
                        { "POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0 },
                        { "TEXCOORD", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 12, D3D11_INPUT_PER_VERTEX_DATA, 0 },
                        { "NORMAL"  , 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, 20, D3D11_INPUT_PER_VERTEX_DATA, 0 },
                    };
                    UINT numElements = ARRAYSIZE( layout );

                    // Create the input layout
                    hr = g_pd3dDevice->CreateInputLayout(   layout, 
                        numElements, 
                        pVSBlob->GetBufferPointer(),
                        pVSBlob->GetBufferSize(), 
                        &parameter.pVertexLayout );

                    pVSBlob->Release();
                    pVSBlob = NULL;
                    if( FAILED( hr ) )
                        return hr;
                }
            }

        case MeshType::MetaBalls :
            {

                UpdateBlobPositions_StateOfMind(10.0f);
                MakeMetaballsMesh();
                
                parameter.PrimitiveTopology     = D3D_PRIMITIVE_TOPOLOGY::D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST;
                parameter.uiMeshVerticesCount   = g_aBlobSimpleVertices.size();
                parameter.uiMeshStride          = sizeof( SimpleVertex );
                parameter.uiMeshOffset          = 0;

                hr = CreateVertexBuffer(&g_aBlobSimpleVertices[0],  parameter.uiMeshVerticesCount, &parameter.pMeshVtxBuffer);
                if( FAILED( hr ) )
                    return hr;

                if(pVSBlob)
                {
                    // Define the input layout
                    D3D11_INPUT_ELEMENT_DESC layout[] =
                    {
                        { "POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0 },
                        { "TEXCOORD", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 12, D3D11_INPUT_PER_VERTEX_DATA, 0 },
                        { "NORMAL"  , 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, 20, D3D11_INPUT_PER_VERTEX_DATA, 0 },
                    };
                    UINT numElements = ARRAYSIZE( layout );

                    // Create the input layout
                    hr = g_pd3dDevice->CreateInputLayout(   layout, 
                        numElements, 
                        pVSBlob->GetBufferPointer(),
                        pVSBlob->GetBufferSize(), 
                        &parameter.pVertexLayout );
                    
                    pVSBlob->Release();
                    pVSBlob = NULL;
                    if( FAILED( hr ) )
                        return hr;
                }
              
                // todo : clarify 
                parameter.uiMeshIndexCount      = g_aBlobTriangles.size() * 3;
                hr = CreateIndexBuffer( &g_aBlobTriangles[0], g_aBlobTriangles.size(), &parameter.pMeshIdxBuffer);
                if( FAILED( hr ) )
                    return hr;

            }
          
        

            break;
        default:
            break;
        }



        it++;
    }

    return hr;
}

HRESULT CreateVertexBuffer(const SimpleVertex* vertices, const size_t size, ID3D11Buffer** ppVertexBuffer) 
{
    D3D11_BUFFER_DESC vertexBufferDesc;
    ZeroMemory( &vertexBufferDesc, sizeof(vertexBufferDesc) );
    
    vertexBufferDesc.Usage            = D3D11_USAGE_DEFAULT;
    vertexBufferDesc.ByteWidth        = sizeof( SimpleVertex ) * size;
    vertexBufferDesc.BindFlags        = D3D11_BIND_VERTEX_BUFFER;
    vertexBufferDesc.CPUAccessFlags   = 0;

    D3D11_SUBRESOURCE_DATA InitData;
    ZeroMemory( &InitData, sizeof(InitData) );
    
    InitData.pSysMem = vertices;
    return g_pd3dDevice->CreateBuffer( &vertexBufferDesc, &InitData, ppVertexBuffer );
}

HRESULT CreateIndexBuffer(const Triangle* indexes, const size_t size, ID3D11Buffer** ppIndexBuffer) 
{
    D3D11_BUFFER_DESC indexBufferDesc;
    ZeroMemory( &indexBufferDesc, sizeof(indexBufferDesc) );

    indexBufferDesc.Usage           = D3D11_USAGE_DEFAULT;
    indexBufferDesc.ByteWidth       = sizeof( Triangle ) * size;
    indexBufferDesc.BindFlags       = D3D11_BIND_INDEX_BUFFER;
    indexBufferDesc.CPUAccessFlags  = 0;
    indexBufferDesc.MiscFlags       = 0;

    D3D11_SUBRESOURCE_DATA InitData;
    ZeroMemory( &InitData, sizeof(InitData) );

    InitData.pSysMem = indexes;
    return g_pd3dDevice->CreateBuffer( &indexBufferDesc, &InitData, ppIndexBuffer );
}





//--------------------------------------------------------------------------------------
// Create Direct3D device and swap chain
//--------------------------------------------------------------------------------------
HRESULT InitDevice()
{
    HRESULT hr = S_OK;

    RECT rc;
    GetClientRect( g_hWnd, &rc );
    UINT width = rc.right - rc.left;
    UINT height = rc.bottom - rc.top;

    UINT createDeviceFlags = 0;
#ifdef _DEBUG
    createDeviceFlags |= D3D11_CREATE_DEVICE_DEBUG;
#endif

    D3D_DRIVER_TYPE driverTypes[] =
    {
        D3D_DRIVER_TYPE_HARDWARE,
        D3D_DRIVER_TYPE_WARP,
        D3D_DRIVER_TYPE_REFERENCE,
    };
    UINT numDriverTypes = ARRAYSIZE( driverTypes );

    D3D_FEATURE_LEVEL featureLevels[] =
    {
        D3D_FEATURE_LEVEL_11_0,
        D3D_FEATURE_LEVEL_10_1,
        D3D_FEATURE_LEVEL_10_0,
    };
	UINT numFeatureLevels = ARRAYSIZE( featureLevels );

    DXGI_SWAP_CHAIN_DESC sd;
    ZeroMemory( &sd, sizeof( sd ) );
    sd.BufferCount = 1;
    sd.BufferDesc.Width = width;
    sd.BufferDesc.Height = height;
    sd.BufferDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
    sd.BufferDesc.RefreshRate.Numerator = 60;
    sd.BufferDesc.RefreshRate.Denominator = 1;
    sd.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
    sd.OutputWindow = g_hWnd;
    sd.SampleDesc.Count = 1;
    sd.SampleDesc.Quality = 0;
    sd.Windowed = TRUE;

    for( UINT driverTypeIndex = 0; driverTypeIndex < numDriverTypes; driverTypeIndex++ )
    {
        g_driverType = driverTypes[driverTypeIndex];
        hr = D3D11CreateDeviceAndSwapChain( NULL, g_driverType, NULL, createDeviceFlags, featureLevels, numFeatureLevels,
                                            D3D11_SDK_VERSION, &sd, &g_pSwapChain, &g_pd3dDevice, &g_featureLevel, &g_pImmediateContext );
        if( SUCCEEDED( hr ) )
            break;
    }
    if( FAILED( hr ) )
        return hr;

    // Create a render target view
    ID3D11Texture2D* pBackBuffer = NULL;
    hr = g_pSwapChain->GetBuffer( 0, __uuidof( ID3D11Texture2D ), ( LPVOID* )&pBackBuffer );
    if( FAILED( hr ) )
        return hr;

    hr = g_pd3dDevice->CreateRenderTargetView( pBackBuffer, NULL, &g_pRenderTargetView );
    pBackBuffer->Release();
    if( FAILED( hr ) )
        return hr;

    g_pImmediateContext->OMSetRenderTargets( 1, &g_pRenderTargetView, NULL );

    // Setup the viewport
    D3D11_VIEWPORT vp;
    vp.Width = (FLOAT)width;
    vp.Height = (FLOAT)height;
    vp.MinDepth = 0.0f;
    vp.MaxDepth = 1.0f;
    vp.TopLeftX = 0;
    vp.TopLeftY = 0;
    g_pImmediateContext->RSSetViewports( 1, &vp );

    float ClearColor[4] = { 0.0f, 0.0f, 0.0f, 1.0f }; //red,green,blue,alpha
    g_pImmediateContext->ClearRenderTargetView( g_pRenderTargetView, ClearColor );
    g_pSwapChain->Present( 0, 0 );


    g_pSoundManager = new SoundManager();
    g_pSoundManager->Init();
   

    SequenceSeed sequences[] = 
    {
   /* { 5.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\ScreenQuadPS_Transition.hlsl"    , ".\\Assets\\animatic1.dds" },
        { 5.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\ScreenQuadPS_Transition.hlsl"    , ".\\Assets\\animatic2.dds" },
        { 5.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\ScreenQuadPS_Transition.hlsl"    , ".\\Assets\\animatic3.dds" },
        { 5.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\ScreenQuadPS_Transition.hlsl"    , ".\\Assets\\animatic4.dds" },
        { 5.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\ScreenQuadPS_Transition.hlsl"    , ".\\Assets\\animatic5.dds" },*/
     //   { 10.0f,  ".\\Src\\TriangleVS.hlsl" , ".\\Src\\DebugPS.hlsl"             , NULL, MeshType::MetaBalls},
        { 15.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\RMSceneStart.hlsl"   , NULL, MeshType::ScreenQuad},//".\\Assets\\animatic6.dds" },
        { 20.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\RMSceneSphere.hlsl"   , NULL, MeshType::ScreenQuad},//".\\Assets\\animatic6.dds" },
        { 20.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\RMSceneMetaBalls.hlsl"   , NULL, MeshType::ScreenQuad},//".\\Assets\\animatic6.dds" },
        { 20.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\RMSceneChurch.hlsl"   , NULL, MeshType::ScreenQuad},//".\\Assets\\animatic6.dds" },
        { 10.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\RmSceneEnd.hlsl"   , NULL, MeshType::ScreenQuad},//".\\Assets\\animatic6.dds" },
 /*        { 5.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\ScreenQuadPS_Transition.hlsl"    , ".\\Assets\\animatic9.dds" },
        { 5.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\ScreenQuadPS_Transition.hlsl"    , ".\\Assets\\animatic10.dds" },
        { 5.0f,  ".\\Src\\SreenQuadVS.hlsl" , ".\\Src\\ScreenQuadPS_Transition.hlsl"    , ".\\Assets\\animatic11.dds" },*/
     };
    
    size_t sequenceCount = std::extent<decltype(sequences)>::value;
    
    g_pSequenceManager  = new SequenceManager();
    g_pSequenceManager->Init(sequences, sequenceCount);

    g_pTimeManager      = new TimeManager(g_pSequenceManager->GetTotalTime());

    g_pSequenceRenderer = new SequenceRenderer();
    g_pSequenceRenderer->Init(sequenceCount);

    hr = LoadResources();

    if( FAILED( hr ) ) 
    {
        MessageBox( NULL,
            L"Load resources failed", L"Error", MB_OK );
        return hr;
    }
	
    // Create the constant buffers
    D3D11_BUFFER_DESC bd;
    ZeroMemory( &bd, sizeof(bd) );
    bd.Usage = D3D11_USAGE_DEFAULT;
    bd.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
    bd.CPUAccessFlags = 0;
    
    bd.ByteWidth = sizeof(CBNeverChanges);
    hr = g_pd3dDevice->CreateBuffer( &bd, NULL, &g_pCBNeverChanges );
    if( FAILED( hr ) )
        return hr;

    bd.ByteWidth = sizeof(CBChangesEveryFrame);
    hr = g_pd3dDevice->CreateBuffer( &bd, NULL, &g_pCBChangesEveryFrame );
    if( FAILED( hr ) )
        return hr;  
    
    bd.ByteWidth = sizeof(CBLedPorperties);
    hr = g_pd3dDevice->CreateBuffer( &bd, NULL, &g_pCBLedPorperties );
    if( FAILED( hr ) )
        return hr;

    ZeroMemory(&g_CBLedPorperties, sizeof(CBLedPorperties));

	// Create the sample state
	D3D11_SAMPLER_DESC sampDesc;
	ZeroMemory( &sampDesc, sizeof(sampDesc) );
	sampDesc.Filter = D3D11_FILTER_MIN_MAG_MIP_LINEAR;
	sampDesc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
	sampDesc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
	sampDesc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
	sampDesc.ComparisonFunc = D3D11_COMPARISON_NEVER;
	sampDesc.MinLOD = 0;
	sampDesc.MaxLOD = D3D11_FLOAT32_MAX;
	hr = g_pd3dDevice->CreateSamplerState( &sampDesc, &g_pSamplerLinear );
	if( FAILED( hr ) )
		return hr;

    // Initialize the view matrix
    XMVECTOR Eye = XMVectorSet( 0.0f, 0.0f, -HalfLedCubeSide, 0.0f );
    XMVECTOR At = XMVectorSet( HalfLedCubeSide, HalfLedCubeSide,HalfLedCubeSide, 0.0f );
    XMVECTOR Up = XMVectorSet( 0.0f, 1.0f, 0.0f, 0.0f );
    g_View = XMMatrixLookAtLH( Eye, At, Up );

    // Initialize the projection matrix
    g_Projection = XMMatrixPerspectiveFovLH( XM_PIDIV2, width / (FLOAT)height, 0.01f, 100.0f );


    g_pSoundManager->PlaySound( "Assets/DemoTrack.xm" );

    return S_OK;
}


//--------------------------------------------------------------------------------------
// Render the frame
//--------------------------------------------------------------------------------------
static bool initiated = false;
void Render()
{
    g_pTimeManager->Tick();
    const int sequenceIndex = g_pSequenceManager->GetSequenceIndex(g_pTimeManager->GetTime(), g_pTimeManager->GetDeltaTime());
    const SequenceSeed  sequence = g_pSequenceManager->GetSeed(sequenceIndex);
    const Interval      interval = g_pSequenceManager->GetInterval(sequenceIndex);


    CBChangesEveryFrame cb;
    //float timerSecSequence = fmod(g_pTimeManager->GetTime(), sequenceIndex == 0 ?  interval.sup : interval.inf);
    float timerSecSequence = g_pTimeManager->GetTime()- interval.inf;
    cb.SequenceRatio  = timerSecSequence / sequence.duration;
    cb.SequenceDuration = sequence.duration;


    //g_pSoundManager->UpdateBuffer(&g_CBLedPorperties.Intensity[0],sizeof(CBLedPorperties) );

    /////////////////////////////////////////////////////////////////////
    //  Metaballs as intensities
    /////////////////////////////////////////////////////////////////////

    if (sequenceIndex == 2)
    {
        UpdateBlobPositions_StateOfMind( 0.5f*timerSecSequence );

    float* pafIntensities = &g_CBLedPorperties.Intensity[0];

    Vector3D vcCorner = g_metaballsTriangulator.GetCorner();
    Vector3D vcIncrement = g_metaballsTriangulator.GetCubeElementSize();
    int nbDivX = g_metaballsTriangulator.GetDivideX();
    int nbDivY = g_metaballsTriangulator.GetDivideY();
    int nbDivZ = g_metaballsTriangulator.GetDivideZ();
    //float thresh = g_objBlobs.GetEnergyThresh();

    int nbDivXY = nbDivX * nbDivY;

    for( int idxZ = 0; idxZ < nbDivZ; ++idxZ )
    {
        g_objBlobs.FuncValuesForPlaneGrid( pafIntensities, vcCorner,
                                           vcIncrement.x, vcIncrement.y,
                                           nbDivX, nbDivY,
                                           NULL );
        for( int i = 0; i < nbDivXY; ++i )
        {
           // now we eant the intensity in [-1;1]
            pafIntensities[i] = g_energyThresholdInv * pafIntensities[i];     
        }

        pafIntensities += nbDivXY;
        vcCorner.z += vcIncrement.z;
    }

    assert( pafIntensities - &g_CBLedPorperties.Intensity[0] <= 4096 );

    }
   
    /* To use the backed cam uncomment here
    // "Interpolate" the plotted camera
    float fCurAnimFrame = timerSecSequence * k_FPS;

    // Avoid overflow
    if ( fCurAnimFrame >= k_framesCount - 1 )
        fCurAnimFrame = float(k_framesCount - 2);
    if ( fCurAnimFrame < 0 )
        fCurAnimFrame = 0;

    // Linearly interpolate between 2 frames from generated data (c.f. AnimationData.cpp)
    int   nCurAnimFrame = (int) floor(fCurAnimFrame);
    float fAnimFrameFract = fCurAnimFrame - nCurAnimFrame;

    cb.CameraPos[0] = k_CamFrames[nCurAnimFrame].pos[0] * (1.f - fAnimFrameFract) + k_CamFrames[nCurAnimFrame + 1].pos[0] * fAnimFrameFract;
    cb.CameraPos[1] = k_CamFrames[nCurAnimFrame].pos[1] * (1.f - fAnimFrameFract) + k_CamFrames[nCurAnimFrame + 1].pos[1] * fAnimFrameFract;
    cb.CameraPos[2] = k_CamFrames[nCurAnimFrame].pos[2] * (1.f - fAnimFrameFract) + k_CamFrames[nCurAnimFrame + 1].pos[2] * fAnimFrameFract;
    cb.CameraPos[3] = 1.f;

    cb.CameraTarget[0] = k_CamFrames[nCurAnimFrame].target[0] * (1.f - fAnimFrameFract) + k_CamFrames[nCurAnimFrame + 1].target[0] * fAnimFrameFract;
    cb.CameraTarget[1] = k_CamFrames[nCurAnimFrame].target[1] * (1.f - fAnimFrameFract) + k_CamFrames[nCurAnimFrame + 1].target[1] * fAnimFrameFract;
    cb.CameraTarget[2] = k_CamFrames[nCurAnimFrame].target[2] * (1.f - fAnimFrameFract) + k_CamFrames[nCurAnimFrame + 1].target[2] * fAnimFrameFract;
    cb.CameraTarget[3] = 1.f;

    cb.CameraRotX = k_CamFrames[nCurAnimFrame].rotx * (1.f - fAnimFrameFract) + k_CamFrames[nCurAnimFrame + 1].rotx * fAnimFrameFract;
    cb.CameraRotY = k_CamFrames[nCurAnimFrame].roty * (1.f - fAnimFrameFract) + k_CamFrames[nCurAnimFrame + 1].roty * fAnimFrameFract;

    // Convert to radians
    cb.CameraRotX *= M_PI / 180.f;
    cb.CameraRotY *= M_PI / 180.f;

    */

    // Just clear the backbuffer
    float ClearColor[4] = { 0.0f, 0.125f, 0.3f, 1.0f }; //red,green,blue,alpha
    g_pImmediateContext->ClearRenderTargetView( g_pRenderTargetView, ClearColor );

    if(!initiated)
    {
        g_pImmediateContext->UpdateSubresource( g_pCBNeverChanges, 0, NULL, &g_cbNeverChange, 0, 0 );
        initiated = true;
    }
   
    g_pImmediateContext->UpdateSubresource( g_pCBChangesEveryFrame, 0, NULL, &cb, 0, 0 );
    g_pImmediateContext->UpdateSubresource( g_pCBLedPorperties, 0, NULL, &g_CBLedPorperties, 0, 0 );
    
    g_pImmediateContext->VSSetConstantBuffers( 0, 1, &g_pCBNeverChanges );
    g_pImmediateContext->VSSetConstantBuffers( 1, 1, &g_pCBChangesEveryFrame );
    g_pImmediateContext->PSSetConstantBuffers( 0, 1, &g_pCBNeverChanges );
    g_pImmediateContext->PSSetConstantBuffers( 1, 1, &g_pCBChangesEveryFrame );
    g_pImmediateContext->PSSetConstantBuffers( 2, 1, &g_pCBLedPorperties );
	g_pImmediateContext->PSSetSamplers( 0, 1, &g_pSamplerLinear );
    
    // Set rendering parameter based on current sequence index	and draw
    g_pSequenceRenderer->SetRenderingParameterAndDraw(sequenceIndex,g_pImmediateContext);

    g_pSwapChain->Present( 0, 0 );
}


//--------------------------------------------------------------------------------------
// Clean up the objects we've created
//--------------------------------------------------------------------------------------
void CleanupDevice()
{
    if( g_pImmediateContext ) g_pImmediateContext->ClearState();

	if( g_pSamplerLinear ) g_pSamplerLinear->Release();
	if( g_pTextureRV ) g_pTextureRV->Release();
    
    if( g_pCBNeverChanges )         g_pCBNeverChanges->Release();
    if( g_pCBChangesEveryFrame )    g_pCBChangesEveryFrame->Release();
	
    if( g_pScreenQuadVertexBuffer ) g_pScreenQuadVertexBuffer->Release();
	if( g_pVertexLayout ) g_pVertexLayout->Release();
	if( g_pVertexShader ) g_pVertexShader->Release();
	if( g_pPixelShader ) g_pPixelShader->Release();

    if( g_pRenderTargetView ) g_pRenderTargetView->Release();
    if( g_pSwapChain ) g_pSwapChain->Release();
    if( g_pImmediateContext ) g_pImmediateContext->Release();
    if( g_pd3dDevice ) g_pd3dDevice->Release();
}

void Shutdown()
{
    if(g_pTimeManager) delete g_pTimeManager;
    if(g_pSequenceManager) delete g_pSequenceManager;
    if(g_pSequenceRenderer) delete g_pSequenceRenderer;
    if(g_pSoundManager) delete g_pSoundManager;
}
